/*****************************************************************************/
/**
@file         FS300_Basic_Measurement.c
@copyright    (c) Rohde & Schwarz GmbH & Co. KG, Munich
@version	   Revision: 1.2, 08/2004                   

@author       W. Blanz
@responsible  W. Blanz
*
@language     C++ (maybe C also)
*
@description	Sample to setup the FS300 Spectrum Analyser
				and perform a measurement. With parameters
				entered  by the user interface.

@see          
*
@history

	05 Dez 2003:	1.0 Initial version
	18 May 2004:	1.1 Added user interface
*
@info
	The example software was tested with the revision 1.8 of the
	rssifs VXI PnP driver and FS300 Release 2.5

*
*  
*
*******************************************************************************
******************************************************************************/


#include "FS300_LabWindowsCVI_Basic_Measurement.h"
#include "rssifs.h"
#include <userint.h>

/* User Interface parameters */
static ViInt32 samplesReturned;		/* returned/set trace samples */
static ViReal64 traceData[2048];	/* trace data buffer */
static ViInt32 sweepPoints;			/* number of points per sweep */
static ViReal64 stopFrequency;		/* stop Frequency of the sweep in Hz */
static ViReal64 startFrequency;		/* start Frequency of the sweep in Hz */
static ViChar message[256];			/* device error/waring messages */
static ViReal64 referenceLevel;		/* reference level */
static ViRsrc resourceName="USB::0x0AAD::0x0006::100196";	 /* device resource string */
static ViStatus Status;				/* device status */
static ViSession instrumentHandle;  /* instrument session handler */
static int panelHandle; 			/* user interface panel handler */


/*======================================================================================================*/
/* Function:	void Error_Message(ViStatus Status)														*/
/* Purpose:		prints visa error and device error to screen											*/
/* Parameter:   ViStatus Status	- Status of last function call											*/
/* Return:		none																					*/
/*======================================================================================================*/
void Error_Message(ViStatus Status)
{
if(Status){ /* display device/driver errors */		
		
	rssifs_error_message (instrumentHandle, Status, &message[0]);
	rssifs_error_query (instrumentHandle, &Status, &message[0]);
	SetCtrlVal (panelHandle, PANEL_STATUS, Status); 
	InsertTextBoxLine (panelHandle, PANEL_ERRBOX, -1, &message[0]);
	ProcessDrawEvents ();
	}	
}


/*======================================================================================================*/
/* Function:	void Basic_Measurement(void)  															*/
/* Purpose:		main measurement function																*/
/* Parameter:   none																					*/
/* Return:		none																					*/
/*======================================================================================================*/
void Basic_Measurement(void)
{

SetCtrlVal (panelHandle, PANEL_RETSWPOINTS, 0);
SetCtrlVal (panelHandle, PANEL_LED, 0);

ProcessDrawEvents ();

GetCtrlVal (panelHandle, PANEL_RESOURCENAME, resourceName);

/* initialise device with reset */
Status = rssifs_init (resourceName, VI_TRUE, VI_TRUE, &instrumentHandle);
Error_Message(Status);

/* Set Reference Level */
GetCtrlVal (panelHandle, PANEL_REFLEVEL, &referenceLevel);
Status = rssifs_confRefLevel (instrumentHandle, referenceLevel);
Error_Message(Status);

/* Set Start and Stop Frequency */
GetCtrlVal (panelHandle, PANEL_STARTFREQ, &startFrequency); 
GetCtrlVal (panelHandle, PANEL_STOPFREQ, &stopFrequency); 
Status = rssifs_confStartStopFrq (instrumentHandle, 0, startFrequency, stopFrequency);
Error_Message(Status);

/* Set RBW and VBW to AUTO Mode */
Status = rssifs_configureBandwidth (instrumentHandle, 0, 0);


/* Stop actual Sweep */
Status = rssifs_actAbort (instrumentHandle); 

/* Set Sweep Points */
GetCtrlVal (panelHandle, PANEL_SWEEPPT, &sweepPoints);
Status = rssifs_confSweepPoints (instrumentHandle, sweepPoints);

/* Start Actual Sweep */
Status = rssifs_actSendTrg (instrumentHandle);

/* Read One Complete Trace */
Status = rssifs_readCompleteSweepData (instrumentHandle, traceData, &samplesReturned);
SetCtrlVal (panelHandle, PANEL_RETSWPOINTS, samplesReturned);
SetCtrlVal (panelHandle, PANEL_LED, 1);
ProcessDrawEvents ();

/* Close Device */
Status = rssifs_close (instrumentHandle);

}


/*======================================================================================================*/
/* Function:	int main(void)				  															*/
/* Purpose:		main function																			*/
/* Parameter:   none																					*/
/* Return:		none																					*/
/*======================================================================================================*/
void main(void)
{
 /* init user interface */
 panelHandle = LoadPanel (0, "FS300_LabWindowsCVI_Basic_Measurement.uir", PANEL);
 DisplayPanel (panelHandle);
 RunUserInterface (); 
}
					  
					  

/* callback function of the OK button */
int CVICALLBACK OkButton (int panel, int control, int event,
		void *callbackData, int eventData1, int eventData2)
{
	switch (event)
		{
		case EVENT_COMMIT:
			/* call device settings function */
			Basic_Measurement();
			break;
		}
	return 0;
}

/* callback function of the Quit button */ 
int CVICALLBACK QuitCallback (int panel, int control, int event,
		void *callbackData, int eventData1, int eventData2)
{
	switch (event)
		{
		case EVENT_COMMIT:
			QuitUserInterface (0);
			return(0);
			break;
		}
	return 0;
}
